<?php
require_once 'config.php';

if(!isLoggedIn()) {
    header("Location: login.php");
    exit();
}

if(!isActiveUser()) {
    header("Location: activate.php");
    exit();
}

$user_id = $_SESSION['user_id'];

// Get categories
$categories = $conn->query("SELECT * FROM categories WHERE id IN (SELECT DISTINCT category_id FROM products WHERE status = 'active') ORDER BY display_order");

// Get products with pagination
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
$per_page = 12;
$offset = ($page - 1) * $per_page;

// Search and filter
$category_filter = isset($_GET['category']) ? sanitize($_GET['category']) : '';
$search = isset($_GET['search']) ? sanitize($_GET['search']) : '';

$where_conditions = ["status = 'active'"];
$params = [];
$types = "";

if($category_filter) {
    $where_conditions[] = "category_id = ?";
    $params[] = $category_filter;
    $types .= "i";
}

if($search) {
    $where_conditions[] = "(name LIKE ? OR description LIKE ?)";
    $params[] = "%$search%";
    $params[] = "%$search%";
    $types .= "ss";
}

$where_sql = !empty($where_conditions) ? "WHERE " . implode(" AND ", $where_conditions) : "";

// Get total products
$count_sql = "SELECT COUNT(*) as total FROM products $where_sql";
$count_stmt = $conn->prepare($count_sql);
if($types) $count_stmt->bind_param($types, ...$params);
$count_stmt->execute();
$total_result = $count_stmt->get_result();
$total_products = $total_result->fetch_assoc()['total'];
$total_pages = ceil($total_products / $per_page);

// Get products
$products_sql = "SELECT p.*, c.name as category_name 
                 FROM products p 
                 LEFT JOIN categories c ON p.category_id = c.id 
                 $where_sql 
                 ORDER BY p.created_at DESC 
                 LIMIT ? OFFSET ?";
$params[] = $per_page;
$params[] = $offset;
$types .= "ii";

$products_stmt = $conn->prepare($products_sql);
$products_stmt->bind_param($types, ...$params);
$products_stmt->execute();
$products_result = $products_stmt->get_result();
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Products - Project Sale</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/animate.css/4.1.1/animate.min.css">
    <style>
        .product-card {
            transition: all 0.3s ease;
        }
        .product-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 20px 40px rgba(0,0,0,0.1);
        }
        .price-badge {
            position: absolute;
            top: 15px;
            right: 15px;
            z-index: 10;
        }
        .category-filter {
            transition: all 0.2s ease;
        }
        .category-filter:hover {
            transform: scale(1.05);
        }
    </style>
</head>
<body class="bg-gray-50">
    <!-- Include Header -->
    <?php include 'includes/header.php'; ?>
    
    <div class="flex min-h-screen">
        <!-- Sidebar -->
        <div class="hidden md:block w-64 bg-white shadow-lg">
            <?php include 'includes/sidebar.php'; ?>
        </div>
        
        <!-- Main Content -->
        <div class="flex-1 p-6">
            <!-- Header -->
            <div class="mb-8">
                <h1 class="text-3xl font-bold text-gray-800 mb-2">
                    <i class="fas fa-shopping-cart mr-2"></i> Our Products
                </h1>
                <p class="text-gray-600">Browse and purchase premium digital products</p>
            </div>
            
            <!-- Search and Filter -->
            <div class="bg-white rounded-xl shadow-lg p-6 mb-8 animate__animated animate__fadeIn">
                <form method="GET" action="" class="space-y-4">
                    <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                        <!-- Search Input -->
                        <div class="relative">
                            <input type="text" name="search" value="<?php echo htmlspecialchars($search); ?>"
                                   class="w-full pl-12 pr-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition duration-300"
                                   placeholder="Search products...">
                            <i class="fas fa-search absolute left-4 top-1/2 transform -translate-y-1/2 text-gray-400"></i>
                        </div>
                        
                        <!-- Category Filter -->
                        <div class="relative">
                            <select name="category" 
                                    class="w-full pl-12 pr-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition duration-300 appearance-none">
                                <option value="">All Categories</option>
                                <?php while($cat = $categories->fetch_assoc()): ?>
                                    <option value="<?php echo $cat['id']; ?>" <?php echo $category_filter == $cat['id'] ? 'selected' : ''; ?>>
                                        <?php echo $cat['name']; ?>
                                    </option>
                                <?php endwhile; ?>
                            </select>
                            <i class="fas fa-filter absolute left-4 top-1/2 transform -translate-y-1/2 text-gray-400"></i>
                            <i class="fas fa-chevron-down absolute right-4 top-1/2 transform -translate-y-1/2 text-gray-400"></i>
                        </div>
                    </div>
                    
                    <!-- Action Buttons -->
                    <div class="flex flex-wrap gap-3">
                        <button type="submit" 
                                class="bg-gradient-to-r from-blue-600 to-indigo-600 text-white font-bold py-3 px-6 rounded-lg hover:from-blue-700 hover:to-indigo-700 transition duration-300 transform hover:scale-105">
                            <i class="fas fa-search mr-2"></i> Search
                        </button>
                        
                        <?php if($search || $category_filter): ?>
                            <a href="products.php" 
                               class="bg-gray-200 text-gray-700 font-bold py-3 px-6 rounded-lg hover:bg-gray-300 transition duration-300">
                                <i class="fas fa-times mr-2"></i> Clear Filters
                            </a>
                        <?php endif; ?>
                    </div>
                </form>
            </div>
            
            <!-- Category Filters -->
            <div class="mb-8">
                <div class="flex items-center mb-4">
                    <h3 class="text-lg font-bold text-gray-700">
                        <i class="fas fa-tags mr-2"></i> Categories
                    </h3>
                    <a href="products.php" class="ml-auto text-blue-600 hover:text-blue-800 text-sm">
                        View All
                    </a>
                </div>
                <div class="grid grid-cols-2 sm:grid-cols-3 md:grid-cols-4 lg:grid-cols-6 gap-3">
                    <a href="products.php" 
                       class="category-filter text-center p-4 bg-white rounded-lg shadow hover:shadow-md transition duration-300 <?php echo !$category_filter ? 'ring-2 ring-blue-500' : ''; ?>">
                        <div class="w-10 h-10 bg-blue-100 rounded-full flex items-center justify-center mx-auto mb-2">
                            <i class="fas fa-boxes text-blue-600"></i>
                        </div>
                        <p class="text-sm font-medium text-gray-700">All</p>
                        <p class="text-xs text-gray-500"><?php echo $total_products; ?> items</p>
                    </a>
                    
                    <?php 
                    $categories->data_seek(0); // Reset pointer
                    while($cat = $categories->fetch_assoc()): 
                        $cat_count = $conn->query("SELECT COUNT(*) as count FROM products WHERE category_id = {$cat['id']} AND status = 'active'")->fetch_assoc()['count'];
                    ?>
                    <a href="products.php?category=<?php echo $cat['id']; ?>" 
                       class="category-filter text-center p-4 bg-white rounded-lg shadow hover:shadow-md transition duration-300 <?php echo $category_filter == $cat['id'] ? 'ring-2 ring-purple-500' : ''; ?>">
                        <div class="w-10 h-10 bg-purple-100 rounded-full flex items-center justify-center mx-auto mb-2">
                            <i class="fas fa-folder text-purple-600"></i>
                        </div>
                        <p class="text-sm font-medium text-gray-700"><?php echo $cat['name']; ?></p>
                        <p class="text-xs text-gray-500"><?php echo $cat_count; ?> items</p>
                    </a>
                    <?php endwhile; ?>
                </div>
            </div>
            
            <!-- Products Grid -->
            <div class="mb-8">
                <div class="flex items-center justify-between mb-6">
                    <h3 class="text-xl font-bold text-gray-800">
                        <i class="fas fa-box-open mr-2"></i> 
                        <?php echo $search ? 'Search Results' : 'Featured Products'; ?>
                        <span class="text-gray-500 text-lg font-normal ml-2">
                            (<?php echo $total_products; ?> products found)
                        </span>
                    </h3>
                    
                    <!-- Sort Options -->
                    <div class="flex items-center space-x-2">
                        <span class="text-gray-600 text-sm">Sort by:</span>
                        <select class="border border-gray-300 rounded-lg px-3 py-2 text-sm focus:ring-2 focus:ring-blue-500 focus:border-blue-500">
                            <option>Newest</option>
                            <option>Price: Low to High</option>
                            <option>Price: High to Low</option>
                            <option>Most Popular</option>
                        </select>
                    </div>
                </div>
                
                <?php if($products_result->num_rows > 0): ?>
                    <div class="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-6">
                        <?php while($product = $products_result->fetch_assoc()): 
                            $discount = 0;
                            if($product['original_price'] > 0 && $product['sale_price'] > 0) {
                                $discount = round((($product['original_price'] - $product['sale_price']) / $product['original_price']) * 100);
                            }
                            $additional_images = !empty($product['additional_images']) ? json_decode($product['additional_images'], true) : [];
                        ?>
                        <div class="product-card bg-white rounded-xl shadow-lg overflow-hidden animate__animated animate__fadeInUp">
                            <!-- Product Image -->
                            <div class="relative h-48 overflow-hidden">
                                <img src="<?php echo $product['image_url'] ?: 'https://via.placeholder.com/300x200?text=Product+Image'; ?>" 
                                     alt="<?php echo htmlspecialchars($product['name']); ?>"
                                     class="w-full h-full object-cover transition duration-300 hover:scale-110">
                                
                                <!-- Discount Badge -->
                                <?php if($discount > 0): ?>
                                <div class="price-badge bg-gradient-to-r from-red-500 to-pink-600 text-white px-3 py-1 rounded-full font-bold shadow-lg">
                                    -<?php echo $discount; ?>%
                                </div>
                                <?php endif; ?>
                                
                                <!-- Quick Actions -->
                                <div class="absolute bottom-3 left-3 right-3 flex justify-between opacity-0 hover:opacity-100 transition duration-300">
                                    <button onclick="addToCart(<?php echo $product['id']; ?>)" 
                                            class="bg-white text-blue-600 p-2 rounded-full shadow hover:bg-blue-50 transition duration-300">
                                        <i class="fas fa-cart-plus"></i>
                                    </button>
                                    <button onclick="quickView(<?php echo $product['id']; ?>)" 
                                            class="bg-white text-purple-600 p-2 rounded-full shadow hover:bg-purple-50 transition duration-300">
                                        <i class="fas fa-eye"></i>
                                    </button>
                                </div>
                            </div>
                            
                            <!-- Product Info -->
                            <div class="p-5">
                                <!-- Category -->
                                <div class="mb-2">
                                    <span class="inline-block px-2 py-1 bg-blue-100 text-blue-700 text-xs font-medium rounded">
                                        <?php echo $product['category_name'] ?: 'Uncategorized'; ?>
                                    </span>
                                </div>
                                
                                <!-- Product Name -->
                                <h4 class="font-bold text-gray-800 text-lg mb-2 truncate" title="<?php echo htmlspecialchars($product['name']); ?>">
                                    <?php echo htmlspecialchars($product['name']); ?>
                                </h4>
                                
                                <!-- Description -->
                                <p class="text-gray-600 text-sm mb-4 line-clamp-2 h-10">
                                    <?php echo strip_tags(substr($product['description'], 0, 80)); ?>...
                                </p>
                                
                                <!-- Price -->
                                <div class="flex items-center justify-between mb-4">
                                    <div>
                                        <?php if($product['original_price'] > 0): ?>
                                            <span class="text-gray-400 line-through text-sm">
                                                ৳<?php echo number_format($product['original_price'], 2); ?>
                                            </span>
                                        <?php endif; ?>
                                        <span class="text-2xl font-bold text-gray-800 ml-2">
                                            ৳<?php echo number_format($product['sale_price'], 2); ?>
                                        </span>
                                    </div>
                                    
                                    <!-- Additional Images Count -->
                                    <?php if(count($additional_images) > 0): ?>
                                    <div class="text-gray-500 text-sm">
                                        <i class="fas fa-images mr-1"></i>
                                        <?php echo count($additional_images); ?>
                                    </div>
                                    <?php endif; ?>
                                </div>
                                
                                <!-- Action Buttons -->
                                <div class="flex space-x-2">
                                    <a href="product-details.php?id=<?php echo $product['id']; ?>" 
                                       class="flex-1 bg-gradient-to-r from-blue-500 to-indigo-600 text-white font-bold py-2 px-4 rounded-lg text-center hover:from-blue-600 hover:to-indigo-700 transition duration-300">
                                        <i class="fas fa-eye mr-1"></i> View
                                    </a>
                                    <button onclick="addToCart(<?php echo $product['id']; ?>)" 
                                            class="flex-1 bg-gradient-to-r from-green-500 to-emerald-600 text-white font-bold py-2 px-4 rounded-lg hover:from-green-600 hover:to-emerald-700 transition duration-300">
                                        <i class="fas fa-cart-plus mr-1"></i> Buy
                                    </button>
                                </div>
                            </div>
                        </div>
                        <?php endwhile; ?>
                    </div>
                    
                    <!-- Pagination -->
                    <?php if($total_pages > 1): ?>
                    <div class="mt-12 flex justify-center">
                        <nav class="flex items-center space-x-2">
                            <?php if($page > 1): ?>
                                <a href="products.php?page=<?php echo $page-1; ?><?php echo $category_filter ? '&category='.$category_filter : ''; ?><?php echo $search ? '&search='.urlencode($search) : ''; ?>"
                                   class="w-10 h-10 flex items-center justify-center rounded-lg bg-white border border-gray-300 hover:bg-gray-50 transition duration-300">
                                    <i class="fas fa-chevron-left"></i>
                                </a>
                            <?php endif; ?>
                            
                            <?php 
                            $start = max(1, $page - 2);
                            $end = min($total_pages, $page + 2);
                            
                            for($i = $start; $i <= $end; $i++): 
                            ?>
                                <a href="products.php?page=<?php echo $i; ?><?php echo $category_filter ? '&category='.$category_filter : ''; ?><?php echo $search ? '&search='.urlencode($search) : ''; ?>"
                                   class="w-10 h-10 flex items-center justify-center rounded-lg font-medium transition duration-300
                                          <?php echo $i == $page ? 'bg-blue-600 text-white' : 'bg-white border border-gray-300 hover:bg-gray-50 text-gray-700'; ?>">
                                    <?php echo $i; ?>
                                </a>
                            <?php endfor; ?>
                            
                            <?php if($page < $total_pages): ?>
                                <a href="products.php?page=<?php echo $page+1; ?><?php echo $category_filter ? '&category='.$category_filter : ''; ?><?php echo $search ? '&search='.urlencode($search) : ''; ?>"
                                   class="w-10 h-10 flex items-center justify-center rounded-lg bg-white border border-gray-300 hover:bg-gray-50 transition duration-300">
                                    <i class="fas fa-chevron-right"></i>
                                </a>
                            <?php endif; ?>
                        </nav>
                    </div>
                    <?php endif; ?>
                    
                <?php else: ?>
                    <!-- No Products Found -->
                    <div class="text-center py-16 bg-white rounded-xl shadow-lg animate__animated animate__fadeIn">
                        <div class="w-24 h-24 bg-gray-100 rounded-full flex items-center justify-center mx-auto mb-6">
                            <i class="fas fa-box-open text-gray-400 text-4xl"></i>
                        </div>
                        <h3 class="text-2xl font-bold text-gray-700 mb-3">No Products Found</h3>
                        <p class="text-gray-600 mb-8 max-w-md mx-auto">
                            <?php echo $search ? 'No products match your search criteria.' : 'No products available at the moment.'; ?>
                        </p>
                        <div class="flex justify-center space-x-4">
                            <a href="products.php" 
                               class="bg-gradient-to-r from-blue-600 to-indigo-600 text-white font-bold py-3 px-6 rounded-lg hover:from-blue-700 hover:to-indigo-700 transition duration-300">
                                <i class="fas fa-redo mr-2"></i> Reset Filters
                            </a>
                            <a href="index.php" 
                               class="bg-gray-200 text-gray-700 font-bold py-3 px-6 rounded-lg hover:bg-gray-300 transition duration-300">
                                <i class="fas fa-home mr-2"></i> Go Home
                            </a>
                        </div>
                    </div>
                <?php endif; ?>
            </div>
            
            <!-- Featured Categories -->
            <div class="bg-gradient-to-r from-blue-50 to-indigo-50 rounded-2xl p-8 mb-8">
                <h3 class="text-2xl font-bold text-gray-800 mb-6 text-center">
                    <i class="fas fa-star mr-2"></i> Popular Categories
                </h3>
                <div class="grid grid-cols-2 md:grid-cols-4 gap-6">
                    <?php 
                    $popular_cats = $conn->query("
                        SELECT c.*, COUNT(p.id) as product_count 
                        FROM categories c 
                        LEFT JOIN products p ON c.id = p.category_id 
                        WHERE p.status = 'active'
                        GROUP BY c.id 
                        ORDER BY product_count DESC 
                        LIMIT 4
                    ");
                    
                    while($cat = $popular_cats->fetch_assoc()): 
                    ?>
                    <a href="products.php?category=<?php echo $cat['id']; ?>" 
                       class="bg-white p-6 rounded-xl shadow hover:shadow-xl transition duration-300 transform hover:scale-105 text-center">
                        <div class="w-16 h-16 bg-gradient-to-r from-blue-500 to-purple-500 rounded-full flex items-center justify-center mx-auto mb-4">
                            <i class="fas fa-folder text-white text-2xl"></i>
                        </div>
                        <h4 class="font-bold text-gray-800 text-lg mb-2"><?php echo $cat['name']; ?></h4>
                        <p class="text-gray-500 text-sm"><?php echo $cat['product_count']; ?> products</p>
                    </a>
                    <?php endwhile; ?>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Quick View Modal -->
    <div id="quickViewModal" class="fixed inset-0 bg-black bg-opacity-50 z-50 hidden items-center justify-center p-4">
        <div class="bg-white rounded-2xl max-w-4xl w-full max-h-[90vh] overflow-hidden">
            <div class="p-4 border-b flex justify-between items-center">
                <h3 class="text-xl font-bold text-gray-800">Quick View</h3>
                <button onclick="closeQuickView()" class="text-gray-500 hover:text-gray-700">
                    <i class="fas fa-times text-xl"></i>
                </button>
            </div>
            <div id="quickViewContent" class="p-6 overflow-y-auto max-h-[calc(90vh-80px)]">
                <!-- Content loaded via AJAX -->
            </div>
        </div>
    </div>
    
    <script>
    function addToCart(productId) {
        fetch('ajax/add-to-cart.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/x-www-form-urlencoded',
            },
            body: 'product_id=' + productId
        })
        .then(response => response.json())
        .then(data => {
            if(data.success) {
                // Show success message
                showNotification('Product added to cart!', 'success');
                
                // Update cart count
                if(data.cart_count) {
                    updateCartCount(data.cart_count);
                }
            } else {
                showNotification(data.message || 'Error adding to cart', 'error');
            }
        })
        .catch(error => {
            console.error('Error:', error);
            showNotification('Network error', 'error');
        });
    }
    
    function quickView(productId) {
        fetch('ajax/quick-view.php?id=' + productId)
            .then(response => response.text())
            .then(html => {
                document.getElementById('quickViewContent').innerHTML = html;
                document.getElementById('quickViewModal').classList.remove('hidden');
                document.getElementById('quickViewModal').classList.add('flex');
            });
    }
    
    function closeQuickView() {
        document.getElementById('quickViewModal').classList.add('hidden');
        document.getElementById('quickViewModal').classList.remove('flex');
    }
    
    function showNotification(message, type) {
        // Remove existing notification
        const existing = document.querySelector('.notification-toast');
        if(existing) existing.remove();
        
        // Create notification
        const notification = document.createElement('div');
        notification.className = `notification-toast fixed top-4 right-4 px-6 py-3 rounded-lg shadow-xl text-white font-bold z-50 animate__animated animate__fadeInRight ${type === 'success' ? 'bg-green-500' : 'bg-red-500'}`;
        notification.innerHTML = `
            <div class="flex items-center">
                <i class="fas ${type === 'success' ? 'fa-check-circle' : 'fa-exclamation-circle'} mr-3 text-xl"></i>
                <span>${message}</span>
            </div>
        `;
        document.body.appendChild(notification);
        
        // Remove after 3 seconds
        setTimeout(() => {
            notification.classList.remove('animate__fadeInRight');
            notification.classList.add('animate__fadeOutRight');
            setTimeout(() => notification.remove(), 300);
        }, 3000);
    }
    
    function updateCartCount(count) {
        // Update cart count in header if exists
        const cartCountEl = document.querySelector('.cart-count');
        if(cartCountEl) {
            cartCountEl.textContent = count;
            cartCountEl.classList.remove('hidden');
        }
    }
    
    // Close modal when clicking outside
    document.addEventListener('click', function(event) {
        const modal = document.getElementById('quickViewModal');
        const modalContent = modal.querySelector('.bg-white');
        
        if(modal.classList.contains('flex') && !modalContent.contains(event.target) && !event.target.closest('button[onclick*="quickView"]')) {
            closeQuickView();
        }
    });
    
    // Add animation to cards
    document.addEventListener('DOMContentLoaded', function() {
        const cards = document.querySelectorAll('.product-card');
        cards.forEach((card, index) => {
            setTimeout(() => {
                card.classList.add('animate__animated', 'animate__fadeInUp');
            }, index * 100);
        });
    });
    </script>
</body>
</html>